<?php

namespace XLite\Module\IPAYINTERNATIONAL\MyPOSVirtual\Model\Payment\Processor;

class MyPOSVirtual extends \XLite\Model\Payment\Base\WebBased
{
    const PAYMENT_METHOD_CARD = 1;
    const PAYMENT_METHOD_IDEAL = 2;
    const PAYMENT_METHOD_ALL = 3;

    const PAYMENT_STATUS_SUCCESS = 1;
    const PAYMENT_STATUS_PENDING = 2;
    const PAYMENT_METHOD_ERROR = 3;
    const PAYMENT_METHOD_EXPIRED = 4;

    public $IPCVersion = '1.4';
    public $IPCLanguage = 'EN';
    public $allowedCurrencies = array('BGN', 'USD', 'EUR', 'GBP', 'CHF', 'JPY', 'RON', 'HRK', 'NOK', 'SEK', 'CZK', 'HUF', 'PLN', 'DKK', 'ISK',);
    public $iDealCurrencies = array('EUR',);
    public $apiConnectTimeout = 120;
    public $learnMoreUrl = 'https://mypos.com/en/ecommerce';
    public $signUpUrl = 'https://www.mypos.com/en/register';
    
    public function __construct()
    {
        parent::__construct();
        
        $this->IPCLanguage = strtoupper(\XLite\Core\Session::getInstance()->getLanguage()->getCode());
    }
    
    /**
     * Get redirect form URL
     *
     * @return string
     */
    protected function getFormURL()
    {
        if ($this->getSetting('mode') != 'live') {
            return $this->getSetting('test_api_url');
        } else {
            return $this->getSetting('live_api_url');
        }
    }

    /**
     * Get redirect form fields list
     *
     * @return array
     */
    protected function getFormFields()
    {
        if ($this->getSetting('mode') != 'live') {
            $sid = $this->getSetting('test_sid');
            $walletNumber = $this->getSetting('test_wallet_nr');
            $keyIndex = $this->getSetting('test_private_key_idx');
            $test_prefix = $this->getSetting('mypos_test_prefix');
        } else {
            $sid = $this->getSetting('live_sid');
            $walletNumber = $this->getSetting('live_wallet_nr');
            $keyIndex = $this->getSetting('live_private_key_idx');
            $test_prefix = null;
        }
        
        $order = $this->transaction->getOrder();
        $currency = $this->transaction->getCurrency();
        $profile = $this->getProfile();
        $email = $profile->getLogin();
        $session = \XLite\Core\Session::getInstance();
        if (!empty($session->checkoutEmail)) {
            $email = $session->checkoutEmail;
            
            // because it won't be unset, nor will the
            // email input be populated with this value
            unset($session->checkoutEmail);
        }
        $billingAddress = $profile->getBillingAddress();
        $formFields = array(
            'IPCmethod' => 'IPCPurchase',
            'KeyIndex' => $keyIndex,
            'IPCVersion' => $this->IPCVersion,
            'IPCLanguage' => $this->IPCLanguage,
            'WalletNumber' => $walletNumber,
            'SID' => $sid,

            'PaymentParametersRequired' => $this->getSetting('payment_parameters_required'),
            'PaymentMethod' => $this->getSetting('payment_method'),

            'Amount' => $currency->roundValue($order->getTotal()),
            'Currency' => $currency->getCode(),
            'OrderID' => (!is_null($test_prefix) ? $test_prefix : '') . $order->getOrderId(),
            'URL_OK' => $this->getReturnURL(null, true),
            'URL_Cancel' => $this->getReturnURL(null, true, true),
            'URL_Notify' => $this->getCallbackURL(null, true),
            
            'CustomerIP' => $_SERVER['REMOTE_ADDR'],
            'CustomerEmail' => $email,
            'CustomerPhone' => $billingAddress->getPhone(),
            'CustomerFirstNames' => $billingAddress->getFirstname(),
            'CustomerFamilyName' => $billingAddress->getLastname(),
            'CustomerCountry' => $billingAddress->getCountry()->getCode3(),
            'CustomerCity' => $billingAddress->getCity(),
            'CustomerZIPCode' => $billingAddress->getZipcode(),
            'CustomerAddress' => $billingAddress->getStreet(),
            'Note' => $order->getNotes(),
            'Source' => 'sc_x-cart ' . \XLite::getInstance()->getVersion() . ' plugin version: v1.1.0'
        );
        
        $orderItems = $order->getItems();
        $orderSurcharges = $order->getSurcharges();
        $formFields['CartItems'] = count($orderItems) + count($orderSurcharges);
        
        $cartItemsIdx = 1;
        foreach ($orderItems as $item) {
            $formFields['Article_' . $cartItemsIdx] = $item->getName();
            $formFields['Price_' . $cartItemsIdx] = $currency->roundValue($item->getNetPrice());
            $formFields['Quantity_' . $cartItemsIdx] = $item->getAmount();
            $formFields['Amount_' . $cartItemsIdx] = $currency->roundValue($item->calculateNetSubtotal());
            $formFields['Currency_' . $cartItemsIdx] = $formFields['Currency'];
            
            $cartItemsIdx++;
        }
        
        foreach ($orderSurcharges as $surcharge) {
            $formFields['Article_' . $cartItemsIdx] = $surcharge->getName();
            $formFields['Price_' . $cartItemsIdx] = $currency->roundValue($surcharge->getValue());
            $formFields['Quantity_' . $cartItemsIdx] = 1;
            $formFields['Amount_' . $cartItemsIdx] = $currency->roundValue($surcharge->getValue());
            $formFields['Currency_' . $cartItemsIdx] = $formFields['Currency'];
            
            $cartItemsIdx++;
        }
        
        $this->signPostData($formFields);
        
        if ($this->getSetting('log_requests')) {
            \XLite\Module\IPAYINTERNATIONAL\MyPOSVirtual\Main::addLog(
                'Purchase with payment card',
                $formFields
            );
        }

        $order->processSucceed();

        return $formFields;
    }
    
    /**
     * Get allowed currencies
     *
     * @param \XLite\Model\Payment\Method $method Payment method
     *
     * @return array
     */
    protected function getAllowedCurrencies(\XLite\Model\Payment\Method $method)
    {
        if ($this->getSetting('payment_method') != self::PAYMENT_METHOD_IDEAL) {
            return $this->allowedCurrencies;
        } else {
            return $this->iDealCurrencies;
        }
    }
    
    /**
     * Get payment method setting by name
     *
     * @param string $name Setting name
     *
     * @return mixed|null
     */
    protected function getSetting($name)
    {
        $result = parent::getSetting($name);

        if (is_null($result)) {
            $method = \XLite\Core\Database::getRepo('XLite\Model\Payment\Method')->findOneBy(array('service_name' => 'MyPOSVirtual'));
            $result = $method
                ? $method->getSetting($name)
                : null;
        }

        return $result;
    }

    /**
     * Process callback
     *
     * @param \XLite\Model\Payment\Transaction $transaction Callback-owner transaction
     *
     * @return void
     */
    public function processCallback(\XLite\Model\Payment\Transaction $transaction)
    {
        parent::processCallback($transaction);
        
        $request = \XLite\Core\Request::getInstance();
        
        if (!$request->isPost()) {
            return;
        }
        
        $logRequests = $this->getSetting('log_requests');
        
        $postData = $request->getPostData();
        
        $validPostData = $this->verifySignedPostData($postData);
        
        if ($logRequests) {
            \XLite\Module\IPAYINTERNATIONAL\MyPOSVirtual\Main::addLog(
                'Request callback received [valid data: ' . var_export($validPostData, true) . ']',
                $postData
            );
        }
        
        if ($validPostData) {
            if ($this->getSetting('mode') != 'live') {
                $sid = $this->getSetting('test_sid');
            } else {
                $sid = $this->getSetting('live_sid');
            }
            
            $order = $transaction->getOrder();
            
            switch ($postData['IPCmethod']) {
                case 'IPCPurchaseNotify':
                    $currency = $transaction->getCurrency();
                    if ($this->getSetting('mode') != 'live'){
                        $orderId = str_replace($this->getSetting('mypos_test_prefix'), '', $postData['OrderID']);
                    } else {
                        $orderId = $postData['OrderID'];
                    }

                    if (($postData['SID'] == $sid)
                    && ($orderId == $order->getOrderId())
                    && ($postData['Amount'] == $currency->roundValue($order->getTotal()))
                    && ($postData['Currency'] == $currency->getCode())
                    ) {
                        $status = $transaction::STATUS_SUCCESS;
                        
                        $label = static::t('ipc_trnref_label');
                        
                        $transaction->setDataCell(
                            'IPC_Trnref',
                            $postData['IPC_Trnref'],
                            $label
                        );
                        
                        $transaction->getOrder()->setDetail(
                            'IPC_Trnref',
                            $postData['IPC_Trnref'],
                            $label
                        );
                        
                        if ($logRequests) {
                            $logData = array(
                                'OrderID' => $postData['OrderID'],
                                'PublicTxnId' => $transaction->getPublicTxnId(),
                                'IPC_Trnref' => $postData['IPC_Trnref']
                            );
                            
                            \XLite\Module\IPAYINTERNATIONAL\MyPOSVirtual\Main::addLog(
                                'Successful payment notification [IPCPurchaseNotify]',
                                $logData
                            );
                        }
                        
                        $result = 'OK';
                        header('Content-Type: text/html');
                        header('Content-Length: ' . strlen($result));
                        echo $result;
                    } else {
                        $status = $transaction::STATUS_FAILED;
                        
                        if ($logRequests) {
                            \XLite\Module\IPAYINTERNATIONAL\MyPOSVirtual\Main::addLog(
                                'Unsuccessful payment notification [IPCPurchaseNotify]',
                                $postData
                            );
                        }
                    }
                    
                    $transaction->setStatus($status);

                    $this->updateInitialBackendTransaction($transaction, $status);

                    $transaction->registerTransactionInOrderHistory('callback');
                    
                    break;
                
                case 'IPCPurchaseRollback':
                    if (($postData['SID'] == $sid)
                    && ($postData['OrderID'] == $order->getOrderId())
                    && ($transaction->getStatus() == $transaction::STATUS_FAILED)
                    ) {
                        $transaction->setStatus($transaction::STATUS_VOID);
                        
                        $this->updateInitialBackendTransaction($transaction, $status);

                        $transaction->registerTransactionInOrderHistory('callback');
                        
                        if ($logRequests) {
                            $logData = array(
                                'OrderID' => $postData['OrderID'],
                                'PublicTxnId' => $transaction->getPublicTxnId(),
                                'IPC_Trnref' => $postData['IPC_Trnref']
                            );
                            
                            \XLite\Module\IPAYINTERNATIONAL\MyPOSVirtual\Main::addLog(
                                'Cancelation of payment notification [IPCPurchaseRollback]',
                                $logData
                            );
                        }
                        
                        $result = 'OK';
                        header('Content-Type: text/html');
                        header('Content-Length: ' . strlen($result));
                        echo $result;
                    }
                    
                    break;
            }
        }
    }
    
    /**
     * Process return
     *
     * @param \XLite\Model\Payment\Transaction $transaction Return-owner transaction
     *
     * @return void
     */
    public function processReturn(\XLite\Model\Payment\Transaction $transaction)
    {
        parent::processReturn($transaction);
        
        $request = \XLite\Core\Request::getInstance();
        
        $logRequests = $this->getSetting('log_requests');
        
        $postData = $request->getPostData();
        
        $validPostData = $this->verifySignedPostData($postData);
        
        if ($logRequests) {
            \XLite\Module\IPAYINTERNATIONAL\MyPOSVirtual\Main::addLog(
                'Request return received [valid data: ' . var_export($validPostData, true) . ']',
                $postData
            );
        }
        
        if ($validPostData) {
            $detailLabel = 'Status';

            if ($request->cancel) {
                $message = static::t('canceled_payment');

                $this->setDetail(
                    'status',
                    $message,
                    $detailLabel
                );
                $transaction->setNote($message);
                $transaction->setStatus($transaction::STATUS_CANCELED);
                $transaction->registerTransactionInOrderHistory();

                if ($logRequests) {
                    \XLite\Module\IPAYINTERNATIONAL\MyPOSVirtual\Main::addLog(
                        'Customer has canceled checkout before completing their payments',
                        $request->getPostData()
                    );
                }
            } else {
                $message = static::t('successful_payment');

                $this->setDetail(
                    'status',
                    $message,
                    $detailLabel
                );

                $transaction->setNote($message);

                if ($logRequests) {
                    \XLite\Module\IPAYINTERNATIONAL\MyPOSVirtual\Main::addLog(
                        'Customer has paid for the order',
                        $request->getPostData()
                    );
                }
            }
        }
    }
    
    /**
     * Update status of backend transaction related to an initial payment transaction
     *
     * @param \XLite\Model\Payment\Transaction $transaction Payment transaction
     * @param string                           $status      Transaction status
     *
     * @return void
     */
    public function updateInitialBackendTransaction(\XLite\Model\Payment\Transaction $transaction, $status)
    {
        $backendTransaction = $transaction->getInitialBackendTransaction();

        if (isset($backendTransaction)) {
            $backendTransaction->setStatus($status);
            $this->saveDataFromRequest($backendTransaction);
        }
    }
    
    /**
     * Get settings widget or template
     *
     * @return string Widget class name or template path
     */
    public function getSettingsWidget()
    {
        return '\XLite\Module\IPAYINTERNATIONAL\MyPOSVirtual\View\Config';
    }
    
    /**
     * Get payment method row checkout template
     *
     * @param \XLite\Model\Payment\Method $method Payment method
     *
     * @return string
     */
    public function getCheckoutTemplate(\XLite\Model\Payment\Method $method)
    {
        return 'modules/IPAYINTERNATIONAL/MyPOSVirtual/checkout/mypos.twig';
    }
    
    /**
     * Check - payment method has enabled test mode or not
     *
     * @param \XLite\Model\Payment\Method $method Payment method
     *
     * @return boolean
     */
    public function isTestMode(\XLite\Model\Payment\Method $method)
    {
        return $method->getSetting('mode') != 'live';
    }

    /**
     * Check - payment method is configured or not
     *
     * @param \XLite\Model\Payment\Method $method Payment method
     *
     * @return boolean
     */
    public function isConfigured(\XLite\Model\Payment\Method $method)
    {

        if (!$method->getSetting('mypos_test_prefix')) {
            $settings = new \XLite\Model\Payment\MethodSetting();

            $settings->setPaymentMethod($method)
                ->setName('mypos_test_prefix')
                ->setValue( uniqid() . '_');

            $method->addSettings($settings);
        }

        if ($method->getSetting('mode') != 'live') {
            return parent::isConfigured($method)
                    && $method->getSetting('test_api_url')
                    && $method->getSetting('test_sid')
                    && $method->getSetting('test_wallet_nr')
                    && $method->getSetting('test_public_cert')
                    && $method->getSetting('test_private_key')
                    && $method->getSetting('test_private_key_idx');
        } else {
            return parent::isConfigured($method)
                    && $method->getSetting('live_api_url')
                    && $method->getSetting('live_sid')
                    && $method->getSetting('live_wallet_nr')
                    && $method->getSetting('live_public_cert')
                    && $method->getSetting('live_private_key')
                    && $method->getSetting('live_private_key_idx');
        }
    }

    /**
     * @return bool
     */
    function isValidForIdeal()
    {
        return in_array(\XLite::getInstance()->getCurrency()->getCode(), $this->iDealCurrencies);
    }

    /**
     * Get payment method icon URL for the checkout page
     *
     * @return string
     */
    public function getCheckoutIconURL()
    {
        $module = $this->getModule();

        if ($this->isValidForIdeal() &&  $this->getSetting('payment_method') == self::PAYMENT_METHOD_ALL) {
            $imageName = 'card_schemes_ideal_no_bg.png';
        } elseif ($this->isValidForIdeal() &&  $this->getSetting('payment_method') == self::PAYMENT_METHOD_IDEAL) {
            $imageName = 'mypos_ideal_no_bg.png';
        } else {
            $imageName = 'card_schemes_no_bg.png';
        }

        $resourcePath = 'modules/' . $module->getAuthor() . '/' . $module->getName() . '/checkout/' .  $imageName;

        return \XLite\Core\Layout::getInstance()->getResourceWebPath($resourcePath);
    }
    
    /**
     * Get payment method admin zone icon URL
     *
     * @param \XLite\Model\Payment\Method $method Payment method
     *
     * @return string
     */
    public function getAdminIconURL(\XLite\Model\Payment\Method $method)
    {
        return true;
    }
    
    /**
     * Get knowledge base page URL
     *
     * @return string
     */
    public function getKnowledgeBasePageURL()
    {
        return $this->learnMoreUrl;
    }

    /**
     * Get URL of the sign up page
     *
     * @param \XLite\Model\Payment\Method $method Payment method
     *
     * @return string
     */
    public function getSignupPageURL(\XLite\Model\Payment\Method $method)
    {
        return $this->signUpUrl;
    }
    
    /**
     * Because the getSetting method requires the
     * transaction property (which is protected) to be set.
     *
     * @param \XLite\Model\Payment\Transaction $transaction
     */
    public function setTransactionProperty(\XLite\Model\Payment\Transaction $transaction)
    {
        if (!$this->transaction) {
            $this->transaction = $transaction;
        }
    }
    
    /**
     * Makes a 'IPCRefund' method call to the myPOS API
     *
     * @param string $trnRef
     * @param string $orderId
     * @param double $amount
     * @param string $currency
     *
     * @return array
     */
    public function makeRefundApiCall($trnRef, $orderId, $amount, $currency)
    {
        if ($this->getSetting('mode') != 'live') {
            $sid = $this->getSetting('test_sid');
            $walletNumber = $this->getSetting('test_wallet_nr');
            $keyIndex = $this->getSetting('test_private_key_idx');
        } else {
            $sid = $this->getSetting('live_sid');
            $walletNumber = $this->getSetting('live_wallet_nr');
            $keyIndex = $this->getSetting('live_private_key_idx');
        }
        
        $methodParams = array(
            'IPCmethod' => 'IPCRefund',
            'KeyIndex' => $keyIndex,
            'IPCVersion' => $this->IPCVersion,
            'IPCLanguage' => $this->IPCLanguage,
            'WalletNumber' => $walletNumber,
            'SID' => $sid,
            
            'IPC_Trnref' => $trnRef,
            'OrderID' => $orderId,
            'Amount' => $amount,
            'Currency' => $currency,
            'Source' => 'sc_x-cart',// . \XLite::getInstance()->getVersion(),
            'OutputFormat' => 'json'
        );
        
        $errorMsg = '';
        $apiResponse = $this->makeApiCall($methodParams, $errorMsg);
        if ($apiResponse === false) {
            return array(
                'Status' => 1,
                'StatusMsg' => $errorMsg
            );
        } else {
            return json_decode($apiResponse, true);
        }
    }

    public function checkPaymentStatusApiCall($orderId)
    {
        if ($this->getSetting('mode') != 'live') {
            $sid = $this->getSetting('test_sid');
            $walletNumber = $this->getSetting('test_wallet_nr');
            $keyIndex = $this->getSetting('test_private_key_idx');
        } else {
            $sid = $this->getSetting('live_sid');
            $walletNumber = $this->getSetting('live_wallet_nr');
            $keyIndex = $this->getSetting('live_private_key_idx');
        }

        $methodParams = array(
            'IPCmethod' => 'IPCGetPaymentStatus',
            'KeyIndex' => $keyIndex,
            'IPCVersion' => $this->IPCVersion,
            'IPCLanguage' => $this->IPCLanguage,
            'WalletNumber' => $walletNumber,
            'SID' => $sid,

            'OrderID' => $orderId,

            'Source' => 'sc_x-cart',
            'OutputFormat' => 'json'
        );

        $errorMsg = '';
        $apiResponse = $this->makeApiCall($methodParams, $errorMsg);
        if ($apiResponse === false) {
            return array(
                'Status' => 1,
                'StatusMsg' => $errorMsg
            );
        } else {
            return json_decode($apiResponse, true);
        }
    }

    /**
     * Make a call to the myPOS API
     *
     * @param array $params
     * @param string &$errorMsg
     *
     * @return string
     */
    private function makeApiCall($params, &$errorMsg)
    {
        if (empty($params['Signature'])) {
            $this->signPostData($params);
        }
        
        $logRequests = $this->getSetting('log_requests');
        
        if ($logRequests) {
            \XLite\Module\IPAYINTERNATIONAL\MyPOSVirtual\Main::addLog(
                'API call',
                $params
            );
        }

        $ch = curl_init();

        curl_setopt_array($ch, array(
            CURLOPT_URL => $this->getFormURL(),
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => $params,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_CONNECTTIMEOUT => $this->apiConnectTimeout,
            CURLOPT_FORBID_REUSE => true,
            CURLOPT_FRESH_CONNECT => true,
            CURLOPT_HEADER => false,
            CURLOPT_USERAGENT => 'X-Cart ' . \XLite::getInstance()->getVersion()
        ));


        $response = curl_exec($ch);

        if ($response === false) {
            $errorMsg = curl_error($ch);
            
            if ($logRequests) {
                \XLite\Module\IPAYINTERNATIONAL\MyPOSVirtual\Main::addLog(
                    'API call error',
                    $errorMsg
                );
            }
        } else {
            if ($logRequests) {
                \XLite\Module\IPAYINTERNATIONAL\MyPOSVirtual\Main::addLog(
                    'API response',
                    $response
                );
            }
        }

        curl_close($ch);

        return $response;
    }
    
    /**
     * Sign the post data, which will be sent to the API
     *
     * @param &array $postData
     */
    private function signPostData(&$postData)
    {
        if ($this->getSetting('mode') != 'live') {
            $privateKey = $this->getSetting('test_private_key');
        } else {
            $privateKey = $this->getSetting('live_private_key');
        }
        
        $tmpPostData = $this->getValuesFromMultidimensionalArray($postData);

        // You need to concatenate all values from $postData and to Base64-encode the result
        $concData = base64_encode(implode('-', $tmpPostData));
        $privKeyObj = openssl_get_privatekey($privateKey);

        // Signed data in binary
        $signature = null;
        openssl_sign($concData, $signature, $privKeyObj, OPENSSL_ALGO_SHA256);

        // Base64 encoding of the signature
        $signature = base64_encode($signature);

        // Now you need to add the signature to the POST request
        $postData['Signature'] = $signature;
        openssl_free_key($privKeyObj);
    }
    
    /**
     * Verify the signature of the received post data
     *
     * @param array $postData
     * @return boolean
     */
    private function verifySignedPostData($postData)
    {
        if (isset($postData['Signature'])) {
            // Save signature
            $signature = $postData['Signature'];

            // Remove signature from POST data array
            unset($postData['Signature']);
        } else {
            // no signature - no success
            return false;
        }
        
        if ($this->getSetting('mode') != 'live') {
            $publicCertificate = $this->getSetting('test_public_cert');
        } else {
            $publicCertificate = $this->getSetting('live_public_cert');
        }

        $postData = $this->getValuesFromMultidimensionalArray($postData);

        // Concatenate all values
        $concData = base64_encode(implode('-', $postData));

        // Extract public key from certificate
        $pubKeyObj = openssl_get_publickey($publicCertificate);

        // Verify signature
        $res = openssl_verify($concData, base64_decode($signature), $pubKeyObj, OPENSSL_ALGO_SHA256);

        openssl_free_key($pubKeyObj);

        if ($res === 1) {
            // success
            return true;
        } else {
            // no success
            return false;
        }
    }
    
    /**
     * Get all values from a multidimensional array
     * and put them in a singledimensional one
     *
     * @param array $array
     * @param array $allValues = array()
     * @return array
     */
    private function getValuesFromMultidimensionalArray($array, $allValues = array())
    {
        foreach ($array as $item) {
            if (is_array($item)) {
                $allValues = $this->getValuesFromMultidimensionalArray($item, $allValues);
            } else {
                $allValues[] = $item;
            }
        }

        return $allValues;
    }
}
